<?php

namespace App\Domain\Core\Actions;

use App\Domain\Core\Models\Branch;
use App\Domain\Order\Models\Order;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class GetAvailableTimesAction
{
    public function __invoke(Branch $branch)
    {
        $currentDate = Carbon::now();
        $startDate = $currentDate;
        $endDate = $startDate->copy()->addWeeks(1);

        $maxOrdersInHour = $branch->max_orders ?? 4;
        $periodPerSession = $branch->period_per_session ?? 60;

        // Define the start and end times for your time slots (adjust as needed)
        $startTime = $branch->start_time ?? '09:00:00';
        $endTime = $branch->end_time ?? '17:00:00';

        $dates = [];

        while ($startDate->lte($endDate)) {
            // Ensure $branch->off_days is not null before decoding
            $offDaysArray = json_decode($branch->off_days, true) ?? [];
            if (!$this->isDayOff($startDate->dayOfWeek, $offDaysArray)) {
                $bookedTimes = $this->getBookedTimes($startDate, $branch->id);
                // Generate all possible time slots for the day
                $timeSlots = [];

                if ($currentDate->isToday()) {
                    if (Carbon::now() > Carbon::parse($startTime)) {
                        $currentTime = Carbon::now()->startOfHour()->addHours(1);
                        $currentDateTime = Carbon::now()->startOfHour()->addHours(1);
                    } else {
                        $currentTime = Carbon::parse($startTime);
                        $currentDateTime = Carbon::parse($startDate->toDateString() . ' ' . $startTime);
                    }

                } else {
                    $currentTime = Carbon::parse($startTime);
                    $currentDateTime = Carbon::parse($startDate->toDateString() . ' ' . $startTime);
                }

                while ($currentTime <= Carbon::parse($endTime)) {
                    $time = $currentTime->format('h:i A');
                    $availableOrders = $maxOrdersInHour - ($bookedTimes[$time] ?? 0);

                    if ($availableOrders > 0) {
                        $timeSlots[] = [
                            'time' => $time,
                            'available_orders' => $availableOrders,
                        ]; // Store available slots and available order count
                    }

                    $currentTime->addMinutes($periodPerSession);
                    $currentDateTime->addMinutes($periodPerSession);
                }

                if (!empty($timeSlots)) {
                    $dates[] = [
                        'date' => $startDate->toDateString(),
                        'time_slots' => $timeSlots,
                    ];
                }
            }
            $startDate->addDay();
        }


        return $dates;
    }

    /**
     * Check if a given day is a day off based on off days array.
     *
     * @param int $dayOfWeek
     * @param array $offDaysArray
     *
     * @return bool
     */
    private function isDayOff(int $dayOfWeek, array $offDaysArray): bool
    {
        return in_array($dayOfWeek, $offDaysArray);
    }

    /**
     * Get booked times for a specific date.
     *
     * @param Carbon $date
     *
     * @return array
     */
    private function getBookedTimes(Carbon $date, $branchId): array
    {
        $bookedTimesData = Order::select(
            DB::raw('TIME_FORMAT(reservation_time, "%h:%i %p") as formatted_time'),
            DB::raw('COUNT(*) as count')
        )
            ->where('reservation_time', $date->toDateString())
            ->where('branch_id', $branchId)
            ->groupBy('formatted_time')
            ->get();

        $bookedTimes = [];
        foreach ($bookedTimesData as $item) {
            $bookedTimes[$item->formatted_time] = $item->count;
        }

        return $bookedTimes;
    }
}
