<?php

namespace App\Http\Controllers\Dashboard\User;

use App\Domain\Core\Enums\CorePermissions;
use App\Domain\Core\Models\Administration\User;
use App\Domain\Core\Models\City;
use App\Domain\User\Datatables\DriverDatatable;
use App\Http\Controllers\Dashboard\DashboardController;
use App\Support\Dashboard\Crud\WithDatatable;
use App\Support\Dashboard\Crud\WithDestroy;
use App\Support\Dashboard\Crud\WithForm;
use App\Support\Dashboard\Crud\WithStore;
use App\Support\Dashboard\Crud\WithUpdate;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Arr;
use Illuminate\Validation\Rules\Password;

class DriverController extends DashboardController
{
    use WithDatatable, WithForm, WithStore, WithUpdate, WithDestroy;

    protected string $name = 'Driver';
    protected string $path = 'dashboard.user.drivers';
    protected string $datatable = DriverDatatable::class;
    protected string $model = User::class;
    protected array $permissions = [CorePermissions::class, 'drivers'];


    protected function rules()
    {
        $rules = [
            'first_name' => "required|string",
            'last_name' => "required|string",
            'lang' => "required|string",
            'lat' => "required|string",
            'lng' => "required|string",
            'address' => "required|string",
            'notify' => "sometimes|nullable|string",
            'identification' => "required|digits:10|unique:users,identification",
            'city_id' => "required|integer|exists:cities,id",
            'status' => "required|string|in:active,pending,banned",
            'phone' => 'required|string|max:15|unique:users,phone',
            'email' => 'required|email|max:255|unique:users,email',
            'password' => ['required', 'confirmed', Password::min(6)],
            'roles' => 'nullable|array',
            'avatar' => 'sometimes|image|max:4000',
            'license_image' => 'sometimes|image|max:4000',
            'company_image' => 'sometimes|image|max:4000',
            'identification_image' => 'sometimes|image|max:2000',
        ];

        if (request()->isMethod('PUT')) {
            $rules['password'] = ['nullable', 'confirmed', Password::min(6)];
            $rules['identification'] = 'required|digits:10|unique:users,identification,' . request()->route('driver');
            $rules['phone'] = 'required|max:255|unique:users,phone,' . request()->route('driver');
            $rules['email'] = 'required|max:255|unique:users,email,' . request()->route('driver');
        }

        return $rules;
    }

    protected function formData(?Model $model = null): array
    {
        return [
            'cities' => toMap(City::get(), 'id', 'title')
        ];
    }

    public function mapDrivers()
    {
        $drivers = User::role('driver')->where('lat', '!=', null)->get();
        return view('dashboard.user.drivers.map-drivers', compact('drivers'));
    }
    public function getDriverLocations()
    {
        $drivers = User::role('driver')->where('lat', '!=', null)->get();
        return response()->json($drivers);
    }

    protected function storeAction(array $validated)
    {
        $avatar = Arr::pull($validated, 'avatar');
        $license_image = Arr::pull($validated, 'license_image');
        $company_image = Arr::pull($validated, 'company_image');
        $identification_image = Arr::pull($validated, 'identification_image');
        $validated['name'] = $validated['first_name'] . ' ' . $validated['last_name'];
        $model = User::create($validated);
        $model->assignRole('driver');

        $avatar && $model->addMedia($avatar)->toMediaCollection('avatar');
        $license_image && $model->addMedia($license_image)
            ->toMediaCollection('license_image');
        $company_image && $model->addMedia($company_image)
            ->toMediaCollection('company_image');
        $identification_image && $model->addMedia($identification_image)
            ->toMediaCollection('identification_image');
    }

    protected function updateAction(array $validated, Model $model)
    {
        $avatar = Arr::pull($validated, 'avatar');
        $license_image = Arr::pull($validated, 'license_image');
        $company_image = Arr::pull($validated, 'company_image');
        $identification_image = Arr::pull($validated, 'identification_image');
        $validated['name'] = $validated['first_name'] . ' ' . $validated['last_name'];

        $model->update($validated);

        if ($avatar instanceof UploadedFile) {
            $model->clearMediaCollection('avatar');
            $model->addMedia($avatar)->toMediaCollection('avatar');
        }
        if ($license_image instanceof UploadedFile) {
            $model->clearMediaCollection('license_image');
            $model->addMedia($license_image)->toMediaCollection('license_image');
        }
        if ($company_image instanceof UploadedFile) {
            $model->clearMediaCollection('company_image');
            $model->addMedia($company_image)->toMediaCollection('company_image');
        }
        if ($identification_image instanceof UploadedFile) {
            $model->clearMediaCollection('identification_image');
            $model->addMedia($identification_image)->toMediaCollection('identification_image');
        }
    }
}
