<?php

namespace App\Domain\Core\Datatables;

use App\Domain\Core\Models\Transaction;
use App\Domain\Order\Models\Order;
use App\Support\Dashboard\Datatables\BaseDatatable;
use Illuminate\Database\Eloquent\Builder;
use Yajra\DataTables\Html\Column;

class TransactionDatatable extends BaseDatatable
{
    protected ?string $actionable = '';


    public function query(): Builder
    {
        return Transaction::query()
            ->when(request('branch_id'), function ($q) {
                $getOrderIDsByBranch = Order::where('branch_id', request('branch_id'))->pluck('id');
                $q->whereIn('trackable_data->order_id',$getOrderIDsByBranch);
            })
            ->when(request('payment_method'), function ($q) {
                $paymentMethod = request('payment_method');
                if(request('payment_method') == 'visa'){
                    $paymentMethod = 'default';
                }
                $q->where('brand','LIKE',"%{$paymentMethod}%");
            })->when(request('order_num'), function ($q) {
                $getOrderId = Order::where('order_num', request('order_num'))->select('id')->first()?->id;
                $q->where('trackable_data->order_id',$getOrderId);
            })->when(request('date'), function ($q) {
                $date = explode(' - ', request('date'));
                $q->whereDate('created_at', '>=', $date[0])
                    ->whereDate('created_at', '<=', $date[1]);
            })
            ->where('status', 'success')
            ->orderBy('created_at', 'desc')->with('user');
    }

    protected function customColumns(): array
    {
        return [
            'created_at' => function ($model) {
                return $model->created_at->translatedFormat('l، j F Y h:i A');
            }, 'payment_brand' => function ($model) {
                return isset(json_decode($model->data,true)['card']) ? json_decode($model->data,true)['paymentBrand'] : $model->data;
            },
        ];
    }

    protected function columns(): array
    {
        return [
            Column::make('checkout_id')->title(__('Checkout id'))->orderable(false),
            Column::make('status')->title(__('Status'))->orderable(false),
            Column::make('amount')->title(__('Transaction Amount'))->orderable(false),
            Column::make('trackable_data')->title(__('Trackable data'))->orderable(false),
            Column::make('payment_brand')->title(__('payment Brand'))->orderable(false),
            Column::make('brand')->title(__('Transaction Brand'))->orderable(false),
            Column::make('user.name')->title(__('Customer Name'))->orderable(false),
            Column::make('user.phone')->title(__('Customer Phone'))->orderable(false),
            Column::make('created_at')->title(__('Created At'))->orderable(false),

        ];
    }
}
