<?php

namespace App\Services;

use GuzzleHttp\Client;
use GuzzleHttp\Middleware;
use GuzzleHttp\Exception\RequestException;
use Illuminate\Support\Facades\File;
use Psr\Http\Message\RequestInterface;
class OpenAITranslationService
{
    protected $client;

    public function __construct()
    {
        // Set unlimited execution time for long-running requests
        ini_set('max_execution_time', 0);

        // Create a Guzzle handler stack with retry logic
        $handlerStack = \GuzzleHttp\HandlerStack::create();
        $handlerStack->push(Middleware::retry(
            function ($retries, RequestInterface $request, $response = null, $exception = null) {
                if ($retries < 3 && ($exception instanceof RequestException || ($response && $response->getStatusCode() >= 500))) {
                    // Log retry details
                    \Log::warning("Retry #{$retries} for " . $request->getUri() . ". Reason: " . ($exception ? $exception->getMessage() : $response->getStatusCode()));
                    return true;
                }
                return false;
            },
            function ($retries) {
                return (int) pow(2, $retries) * 1000; // Exponential backoff
            }
        ));

        // Ensure the OpenAI API key is present
        if (!env('OPENAI_API_KEY')) {
            throw new \InvalidArgumentException('Missing OpenAI API key in environment variables.');
        }

        // Initialize the Guzzle client
        $this->client = new Client([
            'base_uri' => 'https://api.openai.com/v1/',
            'timeout'  => 1200, // Default timeout
            'handler' => $handlerStack,
            'headers' => [
                'Authorization' => 'Bearer ' . env('OPENAI_API_KEY'),
                'Content-Type' => 'application/json',
            ],
        ]);
    }

    private function cleanHtmlOutput($html)
    {
    // Suppress DOM warnings for mis-formed HTML
    libxml_use_internal_errors(true);

    // Create a new DOMDocument; make sure we specify 'UTF-8'
    $dom = new \DOMDocument('1.0', 'UTF-8');

    // Load with an XML preamble to force UTF-8 parsing,
    // and avoid adding <html> or <body> automatically
    $dom->loadHTML(
        '<?xml encoding="UTF-8" ?>' . $html,
        LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD
    );

    // Clear any errors
    libxml_clear_errors();

    // Optionally, if you want to preserve spacing/format:
    // $dom->preserveWhiteSpace = false;
    // $dom->formatOutput = true;

    // Save the DOM back to HTML
    $output = $dom->saveHTML();

    // Remove the <?xml ...  declaration at the top, if present
    $output = preg_replace('/^<\?xml.*?\?>\s*/', '', $output);

    // Decode HTML entities back into real UTF-8 characters
    // so &#12354; becomes あ, etc.
    $output = html_entity_decode($output, ENT_QUOTES | ENT_HTML5, 'UTF-8');

    return $output;
    }
    /**
     * Splits HTML content into multiple chunks without cutting individual top-level nodes.
     *
     * @param string $html       The full HTML content to split.
     * @param int    $maxLength  Approximate max characters per chunk.
     *
     * @return string[]          An array of HTML chunk strings.
     */
    private function splitContent($html, $maxLength = 500)
    {
        // Suppress DOM warnings about malformed HTML
        libxml_use_internal_errors(true);

        // Create a DOMDocument and load HTML
        $dom = new \DOMDocument('1.0', 'UTF-8');

        // Prepend an XML encoding declaration for better UTF-8 handling
        // and prevent DOMDocument from automatically adding <html>, <head>, etc.
        $dom->loadHTML(
            '<?xml encoding="utf-8" ?>' . $html,
            LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD
        );

        // Clear any libxml parsing errors
        libxml_clear_errors();

        // Attempt to locate the <body> element
        $body = $dom->getElementsByTagName('body')->item(0);

        // If no <body> is found, just return the entire HTML as a single chunk
        if (!$body) {
            return [$html];
        }

        $chunks       = [];
        $currentChunk = '';

        // Iterate over each top-level child of <body>
        foreach ($body->childNodes as $childNode) {
            // Convert this node (and its children) into an HTML string
            $outerHTML = $dom->saveHTML($childNode);

            // Check if adding this node would exceed the max length
            if (strlen($currentChunk . $outerHTML) <= $maxLength) {
                $currentChunk .= $outerHTML;
            } else {
                // Start a new chunk if the current one would exceed the limit
                if (!empty($currentChunk)) {
                    $chunks[] = $currentChunk;
                }
                $currentChunk = $outerHTML;
            }
        }

        // Add the last accumulated chunk if not empty
        if (!empty($currentChunk)) {
            $chunks[] = $currentChunk;
        }

        return $chunks;
    }

    public function TranslateArticle($article, string $targetLanguage): array
    {
        \Log::info("=== Starting Translation ===");
        \Log::info("Target Language: " . $targetLanguage);
        try {
            // Language configuration remains the same...
            $languageConfig = [
                'ja' => [
                    'name' => 'Japanese',
                    'chunkSize' => 800,
                    'prompt' => "以下の英語テキストを日本語に翻訳してください：\n\n" .
                        "重要な規則：\n" .
                        "1. []括弧内のテキスト（例：[website_name]）は翻訳しないでください\n" .
                        "2. HTMLタグ（<p>, <h1>など）はそのまま保持してください\n" .
                        "3. href属性のURLは変更しないでください\n" .
                        "4. それ以外のすべてのテキストを自然な日本語に翻訳してください\n\n"
                ],
                'ar' => ['name' => 'Arabic', 'chunkSize' => 1000],
                'fr' => ['name' => 'French', 'chunkSize' => 1000],
                'es' => ['name' => 'Spanish', 'chunkSize' => 1000],
                'pt' => ['name' => 'Portuguese', 'chunkSize' => 1000],
            ];

            // Get language settings or default
            $langSettings = $languageConfig[$targetLanguage] ?? ['name' => 'the target language', 'chunkSize' => 1000];

            $metaDescription =$article->meta_description?? "make your clients happy";

            $titleAndMetaPrompt = <<<PROMPT
                Translate the following title and meta description from English to {$langSettings['name']}.
                
                ⚠️ CRITICAL TRANSLATION RULES:
                1. DO NOT translate ANY text inside square brackets []
                2. Text inside [] MUST stay in English - NOT translated
                3. Examples with [welcome world]:
                   ✓ Correct: "حقق رؤيتك [welcome world]"
                   ✗ Wrong: "حقق رؤيتك [مرحبا بالعالم]"
                4. Keep all HTML tags and URLs unchanged
                
                Title to translate: "{$article->title}"
                Meta description to translate: {$metaDescription}
                PROMPT;

            if ($targetLanguage === 'ja') {
                $titleAndMetaPrompt = "以下のタイトルとメタ記述を日本語に翻訳してください。\n\n" . $titleAndMetaPrompt;
            }

            // Get title and meta description translation
            $titleMetaResponse = $this->client->post('chat/completions', [
                'json' => [
                    'model' => 'gpt-4',
                    'messages' => [
                        [
                            'role' => 'system',
                            'content' => ($targetLanguage === 'ja' ?
                                "あなたは英語から日本語へのプロフェッショナルな翻訳者です。自然な日本語で翻訳してください。" :
                                "You are a professional translator for {$langSettings['name']}. Translate directly and naturally.")
                        ],
                        [
                            'role' => 'user',
                            'content' => $titleAndMetaPrompt
                        ]
                    ],
                    'functions' => [
                        [
                            'name' => 'return_translation',
                            'parameters' => [
                                'type' => 'object',
                                'properties' => [
                                    'title' => [
                                        'type' => 'string',
                                        'description' => 'The translated title'
                                    ],
                                    'meta_description' => [
                                        'type' => 'string',
                                        'description' => 'The translated meta description'
                                    ]
                                ],
                                'required' => ['title', 'meta_description']
                            ]
                        ]
                    ],
                    'function_call' => ['name' => 'return_translation'],
                    'temperature' => 0.3,
                ]
            ]);

            $titleMetaData = json_decode($titleMetaResponse->getBody(), true);
            $titleMetaArgs = json_decode($titleMetaData['choices'][0]['message']['function_call']['arguments'], true);

            $translatedTitle = $titleMetaArgs['title'];
            $translatedMetaDescription = $titleMetaArgs['meta_description'];
            \Log::info("title " . $titleMetaArgs['title']);
            // Now translate the content in chunks
            $chunks = $this->splitContent($article->content, $langSettings['chunkSize']);
            \Log::info("Split into " . count($chunks) . " chunks");

            $translatedContent = '';

            foreach ($chunks as $index => $chunk) {
                \Log::info("Processing chunk " . ($index + 1));

                $contentPrompt = <<<PROMPT
                Translate the following content from English to {$langSettings['name']}.
                
                CRITICAL TRANSLATION RULES:
                1. NEVER MODIFY ANY TEXT INSIDE SQUARE BRACKETS []
                2. Text inside [] must remain EXACTLY as is, including English text
                3. Keep all HTML tags and URLs unchanged
                
                Content to translate:
                {$chunk}
                PROMPT;

                if ($targetLanguage === 'ja') {
                    $contentPrompt = "以下の内容を日本語に翻訳してください。\n\n" . $contentPrompt;
                }

                $response = $this->client->post('chat/completions', [
                    'json' => [
                        'model' => 'gpt-4',
                        'messages' => [
                            [
                                'role' => 'system',
                                'content' => ($targetLanguage === 'ja' ?
                                    "あなたは英語から日本語へのプロフェッショナルな翻訳者です。自然な日本語で翻訳してください。" :
                                    "You are a professional translator for {$langSettings['name']}. Translate directly and naturally.")
                            ],
                            [
                                'role' => 'user',
                                'content' => $contentPrompt
                            ]
                        ],
                        'functions' => [
                            [
                                'name' => 'return_translation',
                                'parameters' => [
                                    'type' => 'object',
                                    'properties' => [
                                        'content' => [
                                            'type' => 'string',
                                            'description' => 'The translated content'
                                        ]
                                    ],
                                    'required' => ['content']
                                ]
                            ]
                        ],
                        'function_call' => ['name' => 'return_translation'],
                        'max_tokens' => 4000,
                        'temperature' => 0.3,
                    ]
                ]);

                $data = json_decode($response->getBody(), true);
                \Log::info("Raw API Response:", ['data' => $data]);

                $functionCallArgsJson = $data['choices'][0]['message']['function_call']['arguments'] ?? '';
                $responseContent = json_decode($functionCallArgsJson, true);

                \Log::info("Parsed Response:", ['content' => $responseContent]);

                if (!$this->validateTranslation($responseContent, $targetLanguage)) {
                    \Log::error("Validation failed for chunk " . ($index + 1));
                    throw new \Exception("Translation validation failed for chunk " . ($index + 1));
                }

                $translatedContent .= $this->cleanHtmlOutput($responseContent['content']);
            }

            return [
                'title' => $translatedTitle,
                'content' => $translatedContent,
                'meta_description' => $translatedMetaDescription,
            ];

        } catch (\Exception $e) {
            \Log::error("Translation failed: " . $e->getMessage());
            return $this->returnOriginal($article);
        }
    }
    private function validateTranslation($response, $targetLanguage): bool
    {
        if (!is_array($response) || empty($response['content'])) {
            \Log::error("Empty or invalid response structure");
            return false;
        }

        // Check for placeholder text that indicates failed translation
        if (strpos($response['content'], '{Translated') !== false) {
            \Log::error("Found placeholder translation text");
            return false;
        }

        $strippedContent = strip_tags($response['content']);

        switch ($targetLanguage) {
            case 'ar':
                if (!preg_match('/\p{Arabic}/u', $strippedContent)) {
                    \Log::error("No Arabic text found in content");
                    return false;
                }
                break;
            case 'fr':
                if (!preg_match('/[éèêëàâîïôûùüçœæ]/i', $strippedContent)) {
                    \Log::error("No French-specific characters found in content");
                    return false;
                }
                break;
            case 'es':
                if (!preg_match('/[áéíóúñ¿¡]/i', $strippedContent)) {
                    \Log::error("No Spanish-specific characters found in content");
                    return false;
                }
                break;
            case 'pt':
                if (!preg_match('/[áéíóúâêôãõç]/i', $strippedContent)) {
                    \Log::error("No Portuguese-specific characters found in content");
                    return false;
                }
                break;
            case 'ja':
                if (!preg_match('/[\p{Hiragana}\p{Katakana}\p{Han}]/u', $strippedContent)) {
                    \Log::error("No Japanese characters found in content");
                    return false;
                }
                break;
        }

        // Additional checks
        $translatedLength = strlen($strippedContent);
        $expectedMinLength = 100; // Adjust based on your content

        if ($translatedLength < $expectedMinLength) {
            \Log::error("Translation seems too short: $translatedLength chars");
            return false;
        }

        \Log::info("Translation validation passed for $targetLanguage");
        return true;
    }

    /**
     * Handle single chunk translation
     */
    private function translateChunk($chunk, $originalTitle, $targetLanguage, $metaDescription, $includeMetadata = true, $temperature = 0): array
    {
        $prompt = $this->buildTranslationPrompt(
            $chunk,
            $originalTitle,
            $targetLanguage,
            $metaDescription,
            $includeMetadata
        );

        $response = $this->client->post('chat/completions', [
            'json' => [
                'model' => 'gpt-4',
                'messages' => [
                    [
                        'role' => 'system',
                        'content' => $targetLanguage === 'ja' ?
                            'You are a professional Japanese translator. Output only valid JSON with no extra text.' :
                            'You are a professional translator. Output only valid JSON with no extra text.'
                    ],
                    [
                        'role' => 'user',
                        'content' => $this->buildTranslationPrompt(
                            $chunk,
                            $originalTitle,
                            $targetLanguage,
                            $metaDescription,
                            $includeMetadata
                        )
                    ]
                ],
                'functions' => [
                    [
                        'name' => 'return_translation',
                        'parameters' => [
                            'type' => 'object',
                            'properties' => [
                                'title' => ['type' => 'string'],
                                'content' => ['type' => 'string'],
                                'meta_description' => ['type' => 'string']
                            ],
                            'required' => ['title', 'content', 'meta_description']
                        ]
                    ]
                ],
                'function_call' => ['name' => 'return_translation'],
                'max_tokens' => 4000,
                'temperature' => $temperature,
            ]
        ]);

        $data = json_decode($response->getBody(), true);

        if (!isset($data['choices'][0]['message']['function_call']['arguments'])) {
            throw new \Exception("Invalid API response structure");
        }

        $result = json_decode($data['choices'][0]['message']['function_call']['arguments'], true);

        if (!isset($result['content']) || empty($result['content'])) {
            throw new \Exception("Empty translation received");
        }

        return [
            'title' => $result['title'] ?? '',
            'content' => $this->cleanHtmlOutput($result['content']),
            'meta_description' => $result['meta_description'] ?? ''
        ];
    }

    /**
     * Build translation prompt
     */
    private function buildTranslationPrompt($content, $title, $targetLanguage, $metaDescription, $includeMetadata): string
    {
        // Special handling for Japanese
        if ($targetLanguage === 'ja') {
            return <<<PROMPT
Translate from English to Japanese (日本語).
Important translation rules:
- Translate to natural, fluent Japanese
- Use appropriate Japanese writing system (Kanji, Hiragana, Katakana)
- Preserve ALL HTML tags and attributes exactly as they are
- Do NOT translate content inside square brackets like [website_title]
- Keep all URLs in href attributes unchanged
- For technical terms, use appropriate Katakana where common in Japanese
- Maintain proper Japanese grammar and sentence structure
- Use appropriate Japanese punctuation (。、etc.)

Title: {$title}
Content: {$content}
meta_description: {$metaDescription}
PROMPT;
        }

        // Default prompt for other languages
        return <<<PROMPT
Translate from en to {$targetLanguage}.
Important rules:
- Preserve ALL HTML tags and attributes exactly as they are
- Do NOT translate anything inside square brackets like [website_title]
- Keep all URLs in href attributes unchanged
- Maintain the same HTML structure
- Ensure high-quality, natural-sounding translation

Title: {$title}
Content: {$content}
meta_description: {$metaDescription}
PROMPT;
    }

    private function returnOriginal($article): array
    {
        \Log::info("Returning original content due to translation failure");
        return [
            'title' => $article['title'],
            'content' => $article['content'],
            'meta_description' => $article['meta_description'] ?? '',
        ];
    }

   /*
    *  private function splitContent($content, $maxTokens = 1500)
    {
        // Split content into manageable chunks
        $chunks = [];
        $words = explode(' ', $content);
        $currentChunk = '';

        foreach ($words as $word) {
            if (strlen($currentChunk . ' ' . $word) > $maxTokens) {
                $chunks[] = $currentChunk;
                $currentChunk = $word;
            } else {
                $currentChunk .= ' ' . $word;
            }
        }

        if (!empty($currentChunk)) {
            $chunks[] = $currentChunk;
        }

        return $chunks;
    }
    */




    public function translateHtml($htmlContent, $targetLanguage = 'en')
    {
        // Split HTML content into manageable chunks
        $dom = new \DOMDocument();
        @$dom->loadHTML($htmlContent, LIBXML_NOERROR | LIBXML_HTML_NODEFDTD | LIBXML_HTML_NOIMPLIED);

        // Translate smaller parts of the HTML separately
        $this->translateTextNodesInChunks($dom, $targetLanguage);

        return $dom->saveHTML();
    }

    protected function translateTextNodesInChunks($node, $targetLanguage)
    {
        foreach ($node->childNodes as $child) {
            if ($child->nodeType === XML_TEXT_NODE) {
                // Split text into smaller chunks for translation
                $chunks = str_split($child->textContent, 500);  // Split into chunks of 500 characters

                $translatedText = '';
                foreach ($chunks as $chunk) {
                    $translatedText .= $this->translate($chunk, $targetLanguage);
                }
                // Replace the original text content with the translated version
                $child->textContent = $translatedText;
            } elseif ($child->hasChildNodes()) {
                $this->translateTextNodesInChunks($child, $targetLanguage);
            }
        }
    }

    public function translate($text, $targetLanguage = 'en')
    {
        $prompt = "Translate the following text directly to {$targetLanguage} if needed. If the text is already in {$targetLanguage}, return it as is. Do not explain the translation, and do not add any extra text, just return the translated text or the original text: " . $text;

        try {

            $response = $this->client->post('chat/completions', [
                'json' => [
                    'model' => 'gpt-4',
                    'messages' => [
                        ['role' => 'system', 'content' => 'You are a helpful assistant who translates text.'],
                        ['role' => 'user', 'content' => $prompt],
                    ],
                    'max_tokens' => 2000,  // Increase if your texts are longer
                ]
            ]);

            $result = json_decode($response->getBody(), true);

            // Return the translated text from the response
            return trim($result['choices'][0]['message']['content']);
        } catch (RequestException $e) {
            // Log or handle the error gracefully
            \Log::error("Error during OpenAI translation: " . $e->getMessage());

            // Optionally return the original text if translation fails
            return $text;
        }
    }

    public function generateTitle($oldTitle,$forTitle = true)
    {

        if($forTitle){

            $prompt = "Rewrite the title '{$oldTitle}' in the same language, ensuring the meaning is preserved. The new title must:
- Retain '\$website_title' and '\$website_name' exactly as they are, without any changes to their position or format .
- Be exactly 35 characters long
- Avoid repeating previously generated titles.
- Return only the new title as plain text, without any single quotation double quotation or, extra characters, or explanations.";

            //$prompt = "detect the language of : ({$oldTitle}) but don't tell me anything about language and Rewrite the following title to have the same meaning with the same language: {$oldTitle} , generate max 15 character , and don't repeat content for next time ,make sure '\$website_title' is always at the beginning of the text, just return the generated text  without double quotation or single quotation, don't remove '\$website_title'. and '\$website_name'";
        }else{

            $prompt = "Rewrite the title '{$oldTitle}' in the same language, ensuring the meaning is preserved. The new title must:
- Retain '\$website_title' and '\$website_name' exactly as they are, without any changes to their position or format .
- Be exactly 135 characters long
- Avoid repeating previously generated titles.
- Return only the new title as plain text, without any single quotation double quotation or, extra characters, or explanations.";
            // $prompt = "detect the language of : ({$oldTitle}) but don't tell me anything about language and Rewrite the following title to have the same meaning with the same language: {$oldTitle} ,generate max 135 character, and don't repeat content for next time ,just return the generated text without double quotation or single quotation,detect var name '\$website_title' keep it in the exact position as it appears in the original text ";
        }
        try {

            $response = $this->client->post('chat/completions', [
                'json' => [
                    'model' => 'gpt-4',
                    'messages' => [
                        ['role' => 'system', 'content' => 'You are a helpful assistant who translates text.'],
                        ['role' => 'user', 'content' => $prompt],
                    ],
                    'max_tokens' => 2000,  // Increase if your texts are longer
                ]
            ]);

            $result = json_decode($response->getBody(), true);

            // Return the translated text from the response
            return trim($result['choices'][0]['message']['content']);
        } catch (RequestException $e) {
            // Log or handle the error gracefully
            \Log::error("Error during OpenAI translation: " . $e->getMessage());

            // Optionally return the original text if translation fails
            return $oldTitle;
        }
    }

    public function generateJsonDescriptionExamplesAndSave($oldDescription, $exampleCount = 25,$lang,$bladeFileName)
    {

        $prompt = "Rewrite the title '{$oldDescription}' while preserving its meaning in the same language. The new titles must follow these guidelines:
- If '\$website_title' and '\$website_name' appear in the original title, retain them exactly as they are, ensuring that there is a space before and after each variable (if applicable).
- If either '\$website_title' or '\$website_name' does not appear in the original title, omit them.
- The total length of the title, excluding '\$website_title' and '\$website_name', must be exactly 135 characters. The length of the variables '\$website_title' and '\$website_name' should not count toward this 135-character limit.
- Ensure that all generated titles are unique, and avoid repetition of previously generated titles.
- Generate a total of {$exampleCount} unique examples only. Do not provide any explanations, clarifications, or other information.
- **Return the results strictly as a JSON array** where each title is a string within the array. Ensure the response is consistent and well-structured.";

        // Make the request to OpenAI API
        try {
            $response = $this->client->post('chat/completions', [
                'json' => [
                    'model' => 'gpt-4',
                    'messages' => [
                        ['role' => 'system', 'content' => 'You are a helpful assistant who generates text examples.'],
                        ['role' => 'user', 'content' => $prompt],
                    ],
                    'max_tokens' => 2000,  // Adjust based on the response length
                ]
            ]);

            // Log the raw response for debugging
            \Log::info($bladeFileName.' in lang '.$lang);
            \Log::info('OpenAI response: ' . $response->getBody());

            // Decode the OpenAI response to get the generated content
            $result = json_decode($response->getBody(), true);

            // Check if the response has valid content
            if (isset($result['choices'][0]['message']['content'])) {
                $content = $result['choices'][0]['message']['content'];

                // Remove extra quotes around the JSON array string
                //$content = trim($content);

                // Check if the content starts and ends with quotes
                if ($content[0] === '"' && $content[strlen($content) - 1] === '"') {
                    //$content = substr($content, 1, -1); // Remove the surrounding quotes
                }

                // Now decode the content as JSON
                $examples = json_decode($content, true);

                // Log the parsed examples for debugging
                \Log::info('Parsed examples: ' . print_r($examples, true));

                // Ensure titles respect the 35-character limit after excluding variables
                if (is_array($examples)) {
                    // Create a unique list of examples (avoiding duplicates)
                    $examples = array_unique($examples);
                    // Define the file path without the time, just the date
                    $filePath = resource_path("views/frontend/{$lang}/json_files/descriptions/{$bladeFileName}.json");

                    // Save the file with the examples
                    $this->SaveFile($filePath, $examples);

                    // Return the path to the saved file
                    return $filePath;
                } else {
                    throw new \Exception("Failed to decode the title examples. Invalid JSON format.");
                }
            } else {
                throw new \Exception("No valid content returned from OpenAI API.");
            }
        } catch (\Exception $e) {
            // Log the error and handle failure gracefully
            \Log::error("Error during OpenAI title generation: " . $e->getMessage());
            return null;
        }
    }

    public function generateJsonTitleExamplesAndSave($oldTitle, $exampleCount = 50, $lang, $bladeFileName)
    {
        // Prepare the prompt for the OpenAI API

        $prompt = "Rewrite the title '{$oldTitle}' while preserving its meaning in the same language. The new titles must follow these guidelines:
- If '\$website_title' and '\$website_name' appear in the original title, retain them exactly as they are, ensuring that there is a space before and after each variable (if applicable).
- If either '\$website_title' or '\$website_name' does not appear in the original title, omit them.
- The total length of the title, excluding '\$website_title' and '\$website_name', must be exactly 35 characters. The length of the variables '\$website_title' and '\$website_name' should not count toward this 35-character limit.
- Ensure that all generated titles are unique, and avoid repetition of previously generated titles.
- Generate a total of {$exampleCount} unique examples only. Do not provide any explanations, clarifications, or other information.
- **Return the results strictly as a JSON array** where each title is a string within the array. Ensure the response is consistent and well-structured.";

        // Make the request to OpenAI API
        try {
            $response = $this->client->post('chat/completions', [
                'json' => [
                    'model' => 'gpt-4',
                    'messages' => [
                        ['role' => 'system', 'content' => 'You are a helpful assistant who generates text examples.'],
                        ['role' => 'user', 'content' => $prompt],
                    ],
                    'max_tokens' => 2000,  // Adjust based on the response length
                ]
            ]);

            // Log the raw response for debugging
            \Log::info($bladeFileName.' in lang '.$lang);
            \Log::info('OpenAI response: ' . $response->getBody());

            // Decode the OpenAI response to get the generated content
            $result = json_decode($response->getBody(), true);

            // Check if the response has valid content
            if (isset($result['choices'][0]['message']['content'])) {
                $content = $result['choices'][0]['message']['content'];

                // Remove extra quotes around the JSON array string
                //$content = trim($content);

                // Check if the content starts and ends with quotes
                if ($content[0] === '"' && $content[strlen($content) - 1] === '"') {
                    //$content = substr($content, 1, -1); // Remove the surrounding quotes
                }

                // Now decode the content as JSON
                $examples = json_decode($content, true);

                // Log the parsed examples for debugging
                \Log::info('Parsed examples: ' . print_r($examples, true));

                // Ensure titles respect the 35-character limit after excluding variables
                if (is_array($examples)) {
                    // Create a unique list of examples (avoiding duplicates)
                    $examples = array_unique($examples);
                    // Define the file path without the time, just the date
                    $filePath = resource_path("views/frontend/{$lang}/json_files/titles/{$bladeFileName}.json");

                    // Save the file with the examples
                    $this->SaveFile($filePath, $examples);

                    // Return the path to the saved file
                    return $filePath;
                } else {
                    throw new \Exception("Failed to decode the title examples. Invalid JSON format.");
                }
            } else {
                throw new \Exception("No valid content returned from OpenAI API.");
            }
        } catch (\Exception $e) {
            // Log the error and handle failure gracefully
            \Log::error("Error during OpenAI title generation: " . $e->getMessage());
            return null;
        }
    }

    public function generateJsonContentExamplesAndSave($oldTitle, $exampleCount = 25, $lang, $bladeFileName)
    {
        $prompt = "Rewrite the title '{$oldTitle}' while preserving its meaning in the same language. The new titles must follow these guidelines:
- If '\$website_title' and '\$website_name' appear in the original title, retain them exactly as they are, ensuring that there is a space before and after each variable (if applicable).
- If either '\$website_title' or '\$website_name' does not appear in the original title, omit them.
- The total length of the title, excluding '\$website_title' and '\$website_name',
- Ensure that all generated titles are unique, and avoid repetition of previously generated titles.
- Generate a total of {$exampleCount} unique examples only. Do not provide any explanations, clarifications, or other information.
- **Return the results strictly as a JSON array** where each title is a string within the array. Ensure the response is consistent and well-structured.";

        try {
            $response = $this->client->post('chat/completions', [
                'json' => [
                    'model' => 'gpt-4',
                    'messages' => [
                        ['role' => 'system', 'content' => 'You are a helpful assistant who generates text examples. Always return responses in valid JSON array format.'],
                        ['role' => 'user', 'content' => $prompt],
                    ],
                    'max_tokens' => 2000,
                    'temperature' => 0.7, // Add some randomness but not too much
                ]
            ]);

            \Log::info("{$bladeFileName} in lang {$lang}");

            $responseBody = $response->getBody()->getContents();
            \Log::info('OpenAI response: ' . $responseBody);

            $result = json_decode($responseBody, true);

            if (!isset($result['choices'][0]['message']['content'])) {
                throw new \Exception("Invalid response structure from OpenAI API");
            }

            $content = $result['choices'][0]['message']['content'];

            // Clean up the content
            $content = $this->cleanJsonContent($content);

            // Parse the JSON content
            $examples = $this->parseJsonContent($content);

            if (!is_array($examples)) {
                throw new \Exception("Failed to parse examples as array");
            }

            // Process and validate examples
            $examples = $this->processExamples($examples, $oldTitle);
            $filePath = resource_path("views/frontend/{$lang}/json_files/change_by_command/{$bladeFileName}.json");

            // Save to file
            $this->SaveFile($filePath, $examples);


            return $filePath;

        } catch (\Exception $e) {
            \Log::error("Error during OpenAI title generation: " . $e->getMessage());
            \Log::error("Trace: " . $e->getTraceAsString());
            return null;
        }
    }

    private function cleanJsonContent($content)
    {
        // Remove any leading/trailing whitespace
        $content = trim($content);

        // If content is wrapped in backticks, remove them
        $content = preg_replace('/^```json\s*|\s*```$/i', '', $content);

        // Handle truncated JSON
        if (!$this->isValidJson($content)) {
            // Try to find the last complete array element
            $content = preg_replace('/,\s*"[^"]*$/', '', $content);
            // Close the array if needed
            if (!str_ends_with(trim($content), ']')) {
                $content = rtrim($content, ',') . ']';
            }
        }

        return $content;
    }

    private function parseJsonContent($content)
    {
        $examples = json_decode($content, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new \Exception("JSON decode error: " . json_last_error_msg());
        }

        return $examples;
    }

    private function processExamples($examples, $oldTitle)
    {
        // Remove duplicates
        $examples = array_unique($examples);

        // Filter out invalid entries
        $examples = array_filter($examples, function($example) {
            return is_string($example) && !empty(trim($example));
        });

        // Ensure variables are properly preserved
        $examples = array_map(function($example) use ($oldTitle) {
            // Preserve $website_title and $website_name if they were in original
            if (strpos($oldTitle, '$website_title') !== false && strpos($example, '$website_title') === false) {
                $example = '$website_title ' . $example;
            }
            if (strpos($oldTitle, '$website_name') !== false && strpos($example, '$website_name') === false) {
                $example = '$website_name ' . $example;
            }
            return $example;
        }, $examples);

        return array_values($examples); // Reindex array
    }



    private function isValidJson($string) {
        json_decode($string);
        return json_last_error() === JSON_ERROR_NONE;
    }
    public function generateForContent($oldContent){
        $prompt = "Rewrite the title '{$oldContent}' in the same language, ensuring the meaning is preserved. The new content must:
- Retain '\$website_title' and '\$website_name' exactly as they are, without any changes to their position or format .
- Avoid repeating previously generated titles.
- Return only the new title as plain text, without any single quotation double quotation or, extra characters, or explanations.";

        //$prompt = "detect the language of : ({$oldContent}) but don't tell me anything about language and Rewrite the following title to have the same meaning with the same language: {$oldContent} , and don't repeat content for next time ,don't remove tags ,just return the generated text without double quotation or single quotation,detect var name '\$website_title' if exists and keep it in the exact position as it appears in the original text ";
        try {

            $response = $this->client->post('chat/completions', [
                'json' => [
                    'model' => 'gpt-4',
                    'messages' => [
                        ['role' => 'system', 'content' => 'You are a helpful assistant who translates text.'],
                        ['role' => 'user', 'content' => $prompt],
                    ],
                    'max_tokens' => 2000,  // Increase if your texts are longer
                ]
            ]);

            $result = json_decode($response->getBody(), true);

            // Return the translated text from the response
            return trim($result['choices'][0]['message']['content']);
        } catch (RequestException $e) {
            // Log or handle the error gracefully
            \Log::error("Error during OpenAI translation: " . $e->getMessage());

            // Optionally return the original text if translation fails
            return $oldTitle;
        }
    }

    /**
     * @param string $filePath
     * @param array $examples
     * @return void
     */
    protected function SaveFile(string $filePath, array $examples): void
    {
// Check if the file exists
        if (File::exists($filePath)) {
            // If the file exists, read its contents and decode the existing JSON
            $existingData = json_decode(file_get_contents($filePath), true);

            // Check if existing data is valid and an array
            if (is_array($existingData)) {
                // Merge the new examples with the existing data
                $examples = array_merge($existingData, $examples);
            }
        }

        // Check if the directory exists, if not, create it
        $directoryPath = dirname($filePath); // Get the directory part of the file path

        if (!File::exists($directoryPath)) {
            // Create the directory and any necessary subdirectories
            File::makeDirectory($directoryPath, 0777, true);
        }
        // Save the updated examples (merged with existing ones if applicable) to the JSON file
        file_put_contents($filePath, json_encode($examples, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    }
}
